/**
 * AGSPA-Level Professional Animations
 * Fast, smooth animations completing within 3 seconds
 */

(function() {
    'use strict';

    // Animation Configuration
    const config = {
        duration: 3000, // Total animation time: 3 seconds
        staggerDelay: 100, // Delay between staggered elements
        easing: 'cubic-bezier(0.4, 0, 0.2, 1)',
        threshold: 0.1,
        once: true
    };

    // Initialize animations on DOM ready
    document.addEventListener('DOMContentLoaded', initAnimations);

    function initAnimations() {
        // Add initial animation class to body
        document.body.classList.add('animations-init');

        // Animate hero section immediately
        animateHero();

        // Setup scroll-triggered animations
        setupScrollAnimations();

        // Setup parallax effects
        setupParallax();

        // Setup hover animations
        setupHoverAnimations();
    }

    // Hero Section Animation - Staggered reveal within 3 seconds
    function animateHero() {
        const heroContainer = document.querySelector('.hero-container');
        if (!heroContainer) return;

        // Elements to animate in hero
        const animatableElements = [
            '.logo',
            '.nav-links',
            '.nav-right',
            '.badge-trusted',
            '.hero-left h1',
            '.hero-subtext',
            '.search-container',
            '.hero-right img'
        ];

        const startTime = performance.now();
        const totalDuration = 2800; // Complete within 2.8 seconds (leaving 0.2s buffer)

        animatableElements.forEach((selector, index) => {
            const element = document.querySelector(selector);
            if (!element) return;

            // Stagger each element
            const delay = index * 150; // 150ms stagger

            setTimeout(() => {
                element.style.opacity = '0';
                element.style.transform = getTransformForElement(selector);
                element.style.transition = `opacity 0.6s ${config.easing}, transform 0.6s ${config.easing}`;

                // Force reflow
                element.offsetHeight;

                element.style.opacity = '1';
                element.style.transform = 'translate(0, 0) scale(1)';
            }, delay);
        });

        // Animate cloud floor
        const cloudFloor = document.querySelector('.cloud-floor');
        if (cloudFloor) {
            setTimeout(() => {
                cloudFloor.style.animation = 'cloudFloat 3s ease-in-out infinite';
            }, 500);
        }
    }

    // Get appropriate transform based on element type
    function getTransformForElement(selector) {
        if (selector.includes('h1') || selector.includes('h2')) {
            return 'translateY(30px) scale(0.95)';
        }
        if (selector.includes('badge')) {
            return 'translateY(-20px)';
        }
        if (selector.includes('search') || selector.includes('subtext')) {
            return 'translateY(20px)';
        }
        if (selector.includes('img')) {
            return 'translateX(50px) scale(0.9)';
        }
        return 'translateY(20px)';
    }

    // Setup scroll-triggered animations
    function setupScrollAnimations() {
        const animatedElements = document.querySelectorAll([
            '.how-card',
            '.esim-card',
            '.feature-card',
            '.post-card',
            '.faq-item',
            '.destination-card',
            '.side-item',
            '.section-title-center h2',
            '.section-title-center p',
            '.why-header h2',
            '.how-left h2',
            '.how-left p',
            '.download-container h2',
            '.download-container p',
            '.refer-text-block h2',
            '.refer-text-block p',
            '.posts-header h2',
            '.footer-brand',
            '.footer-links',
            '.footer-newsletter'
        ].join(', '));

        const observerOptions = {
            root: null,
            rootMargin: '0px',
            threshold: config.threshold
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach((entry, index) => {
                if (config.once && entry.target.classList.contains('animated')) return;

                if (entry.isIntersecting) {
                    const delay = config.once ? 0 : index * 50;
                    setTimeout(() => {
                        animateElement(entry.target);
                    }, delay);

                    if (config.once) {
                        entry.target.classList.add('animated');
                    }
                }
            });
        }, observerOptions);

        animatedElements.forEach(el => {
            el.style.opacity = '0';
            el.style.transform = 'translateY(40px)';
            el.style.transition = `opacity 0.6s ${config.easing}, transform 0.6s ${config.easing}`;
            observer.observe(el);
        });
    }

    // Animate individual element
    function animateElement(element) {
        element.style.opacity = '1';
        element.style.transform = 'translateY(0)';
    }


    // Setup hover animations for interactive elements
    function setupHoverAnimations() {
        // Cards hover effect
        const cards = document.querySelectorAll('.how-card, .esim-card, .feature-card, .post-card');
        
        cards.forEach(card => {
            card.style.transition = 'transform 0.3s ease, box-shadow 0.3s ease';
            
            card.addEventListener('mouseenter', () => {
                card.style.transform = 'translateY(-8px) scale(1.02)';
                card.style.boxShadow = '0 20px 40px rgba(0, 0, 0, 0.15)';
            });

            card.addEventListener('mouseleave', () => {
                card.style.transform = 'translateY(0) scale(1)';
                card.style.boxShadow = '';
            });
        });

        // Button hover animations
        const buttons = document.querySelectorAll('.btn-app, .esim-btn, .store-btn, .nav-btn');
        
        buttons.forEach(btn => {
            btn.style.transition = 'transform 0.2s ease, background-color 0.2s ease';
            
            btn.addEventListener('mouseenter', () => {
                btn.style.transform = 'scale(1.05)';
            });

            btn.addEventListener('mouseleave', () => {
                btn.style.transform = 'scale(1)';
            });
        });

        // Nav links hover
        const navLinks = document.querySelectorAll('.nav-links a');
        
        navLinks.forEach(link => {
            link.style.transition = 'color 0.2s ease, transform 0.2s ease';
            
            link.addEventListener('mouseenter', () => {
                link.style.transform = 'translateY(-2px)';
            });

            link.addEventListener('mouseleave', () => {
                link.style.transform = 'translateY(0)';
            });
        });
    }

    // Floating animation for hero image
    // function addFloatingAnimation() {
    //     const heroImg = document.querySelector('.hero-right img');
    //     if (heroImg) {
    //         heroImg.style.animation = 'floatImage 4s ease-in-out infinite';
    //     }
    // }

    // Add CSS keyframes dynamically
    function addAnimationStyles() {
        const style = document.createElement('style');
        style.textContent = `
            @keyframes floatImage {
                0%, 100% { transform: translateY(0); }
                50% { transform: translateY(-15px); }
            }

            @keyframes cloudFloat {
                0%, 100% { transform: translateX(0); }
                25% { transform: translateX(10px); }
                75% { transform: translateX(-10px); }
            }

            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }

            @keyframes scaleIn {
                from {
                    opacity: 0;
                    transform: scale(0.9);
                }
                to {
                    opacity: 1;
                    transform: scale(1);
                }
            }

            @keyframes slideInLeft {
                from {
                    opacity: 0;
                    transform: translateX(-30px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }

            @keyframes slideInRight {
                from {
                    opacity: 0;
                    transform: translateX(30px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }

            @keyframes rotateIn {
                from {
                    opacity: 0;
                    transform: rotate(-10deg) scale(0.9);
                }
                to {
                    opacity: 1;
                    transform: rotate(0) scale(1);
                }
            }

            @keyframes shimmer {
                0% { background-position: -200% 0; }
                100% { background-position: 200% 0; }
            }

            @keyframes pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.05); }
            }

            @keyframes borderGlow {
                0%, 100% { box-shadow: 0 0 5px rgba(0, 183, 255, 0.3); }
                50% { box-shadow: 0 0 20px rgba(0, 183, 255, 0.6); }
            }

            .animations-init {
                overflow-x: hidden;
            }

            .animated {
                animation-play-state: running !important;
            }
        `;
        document.head.appendChild(style);
    }

    // Initialize styles
    addAnimationStyles();
    addFloatingAnimation();

    // Smooth reveal on page load for faster perceived performance
    window.addEventListener('load', () => {
        document.body.style.opacity = '1';
    });

    // Set initial body opacity for fade-in effect
    document.body.style.opacity = '0';
    document.body.style.transition = 'opacity 0.5s ease';

})();
